#include <sys/types.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include "sys_net.h"
#if !defined(__WIN32__)
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <netdb.h>
#include <sys/wait.h>
#endif
#include "hxd.h"
#include "htxf.h"
#include "rcv.h"
#include "xmalloc.h"

#ifdef CONFIG_COMPRESS
#include "compress.h"
#endif

#ifdef SOCKS
#include "socks.h"
#endif

#if defined(CONFIG_HTXF_PTHREAD)
#include <pthread.h>
#endif

#define HTXF_THREADS_LISTEN	0

u_int16_t nhtlc_conns = 0;

struct htlc_conn __htlc_list, *htlc_list = &__htlc_list, *htlc_tail = &__htlc_list;
#ifdef CONFIG_EXEC
extern void exec_close_all (struct htlc_conn *htlc);
#endif

void
snd_strerror (struct htlc_conn *htlc, int err)
{
	char *str = strerror(err);
	u_int16_t len = strlen(str);

	hlwrite(htlc, HTLS_HDR_TASK, 1, 1, HTLS_DATA_TASKERROR, len, str);
}

void
snd_errorstr (struct htlc_conn *htlc, const char *str)
{
	u_int16_t len = strlen(str);

	hlwrite(htlc, HTLS_HDR_TASK, 1, 1, HTLS_DATA_TASKERROR, len, str);
}

#define READ_BUFSIZE	0x4000
extern unsigned int decode (struct htlc_conn *htlc);

static void
htlc_read (int fd)
{
	ssize_t r;
	struct htlc_conn *htlc = hxd_files[fd].conn.htlc;
	struct qbuf *in = &htlc->read_in;
#if defined(__WIN32__)
	int wsaerr;
#endif

	if (!in->len) {
		qbuf_set(in, in->pos, READ_BUFSIZE);
		in->len = 0;
	}
	r = socket_read(fd, &in->buf[in->pos], READ_BUFSIZE-in->len);
#if defined(__WIN32__)
	wsaerr = WSAGetLastError();
	if (r == 0 || (r < 0 && wsaerr != WSAEWOULDBLOCK && wsaerr != WSAEINTR)) {
#else
	if (r == 0 || (r < 0 && errno != EWOULDBLOCK && errno != EINTR)) {
#endif
		/*hxd_log("htlc_read; %d %s", r, strerror(errno));*/
		htlc_close(htlc);
	} else {
		in->len += r;
		while (decode(htlc)) {
			if (htlc->rcv) {
				int is_rcv_hdr = htlc->rcv == rcv_hdr;
				htlc->rcv(htlc);
				if (!hxd_files[fd].conn.htlc) {
					return;
				}
				if (!is_rcv_hdr)
					goto reset;
			} else {
reset:
				if (!htlc->access_extra.can_login)
					test_away(htlc);
				htlc->rcv = rcv_hdr;
				qbuf_set(&htlc->in, 0, SIZEOF_HL_HDR);
			}
		}
	}
}

static void
htlc_write (int fd)
{
	ssize_t r;
	struct htlc_conn *htlc = hxd_files[fd].conn.htlc;
#if defined(__WIN32__)
	int wsaerr;
#endif

	if (htlc->out.len == 0) {
		/*hxd_log("htlc->out.len == 0 but htlc_write was called...");*/
		hxd_fd_clr(fd, FDW);
		return;
	}
	r = socket_write(fd, &htlc->out.buf[htlc->out.pos], htlc->out.len);
#if defined(__WIN32__)
	wsaerr = WSAGetLastError();
	if (r == 0 || (r < 0 && wsaerr != WSAEWOULDBLOCK && wsaerr != WSAEINTR)) {
#else
	if (r == 0 || (r < 0 && errno != EWOULDBLOCK && errno != EINTR)) {
#endif
		/*hxd_log("htlc_write(%u); %d %s", htlc->out.len, r, strerror(errno));*/
		htlc_close(htlc);
	} else {
		htlc->out.pos += r;
		htlc->out.len -= r;
		if (!htlc->out.len) {
			htlc->out.pos = 0;
			htlc->out.len = 0;
			hxd_fd_clr(fd, FDW);
		}
	}
}

extern void ident_close (int fd);

static int
login_timeout (struct htlc_conn *htlc)
{
	if (hxd_cfg.options.ident && htlc->identfd != -1) {
		ident_close(htlc->identfd);
		return 1;
	}
	if (htlc->access_extra.can_login)
		htlc_close(htlc);

	return 0;
}

static void
listen_ready_read (int fd)
{
	int s;
	struct SOCKADDR_IN saddr;
	int siz = sizeof(saddr);
#ifdef CONFIG_IPV6
	char buf[HOSTLEN+1];
#else
	char buf[16];
#endif
	struct htlc_conn *htlc;

	s = accept(fd, (struct SOCKADDR *)&saddr, &siz);
	if (s < 0) {
		hxd_log("htls: accept: %s", strerror(errno));
		return;
	}
	nr_open_files++;
	if (nr_open_files >= hxd_open_max) {
		hxd_log("%s:%d: %d >= hxd_open_max (%d)", __FILE__, __LINE__, s, hxd_open_max);
		socket_close(s);
		nr_open_files--;
		return;
	}
	fd_closeonexec(s, 1);
	socket_blocking(s, 0);
#ifdef CONFIG_IPV6
	inet_ntop(AFINET, (char *)&saddr.SIN_ADDR, buf, sizeof(buf));
#else
	inet_ntoa_r(saddr.SIN_ADDR, buf, sizeof(buf));
#endif
	hxd_log("%s:%u -- htlc connection accepted", buf, ntohs(saddr.SIN_PORT));

	htlc = xmalloc(sizeof(struct htlc_conn));
	memset(htlc, 0, sizeof(struct htlc_conn));

	htlc->sockaddr = saddr;

	hxd_files[s].ready_read = htlc_read;
	hxd_files[s].ready_write = htlc_write;
	hxd_files[s].conn.htlc = htlc;

	htlc->fd = s;
	htlc->rcv = rcv_magic;
	htlc->trans = 1;
	htlc->chattrans = 1;
	htlc->put_limit = hxd_cfg.limits.individual_uploads > HTXF_PUT_MAX ? HTXF_PUT_MAX : hxd_cfg.limits.individual_uploads;
	htlc->get_limit = hxd_cfg.limits.individual_downloads > HTXF_GET_MAX ? HTXF_GET_MAX : hxd_cfg.limits.individual_downloads;
	htlc->limit_out_Bps = hxd_cfg.limits.out_Bps;
	htlc->limit_uploader_out_Bps = hxd_cfg.limits.uploader_out_Bps;
	INITLOCK_HTXF(htlc);

	if (high_fd < s)
		high_fd = s;

	htlc->flags.visible = 1;

	htlc->identfd = -1;
	if (check_banlist(htlc))
		return;
	htlc->access_extra.can_login = 1;
	timer_add_secs(7, login_timeout, htlc);

#if 0
	if (hxd_cfg.options.hostname_lookup) {
		start_hostname_lookup(htlc);
	}
#endif
	if (hxd_cfg.options.ident) {
		start_ident(htlc);
	} else {
		qbuf_set(&htlc->in, 0, HTLC_MAGIC_LEN); 
		FD_SET(s, &hxd_rfds);
	}
}

#if defined(CONFIG_HTXF_CLONE) || defined(CONFIG_HTXF_PTHREAD)
extern void xxx_remove (struct htxf_conn *htxf);
#endif

#if defined(__WIN32__)
#define pthread_kill(t,s) pthread_cancel(t)
#endif
#if defined(NO_PTHREAD_KILL)
#define pthread_kill(t,s) kill(t,s)
#endif

void
htlc_close (struct htlc_conn *htlc)
{
	int fd = htlc->fd;
#ifdef CONFIG_IPV6
	char buf[HOSTLEN+1];
#else
	char buf[16];
#endif
	int wr;
	u_int16_t i, uid16;
	struct htlc_conn *htlcp;
	int can_login;
#if defined(CONFIG_HTXF_CLONE) || defined(CONFIG_HTXF_FORK)
	int status;
#endif

	socket_close(fd);
	nr_open_files--;
	hxd_fd_clr(fd, FDR|FDW);
	hxd_fd_del(fd);
	memset(&hxd_files[fd], 0, sizeof(struct hxd_file));
	if (htlc->identfd != -1) {
		int ifd = htlc->identfd;
		socket_close(ifd);
		nr_open_files--;
		hxd_fd_clr(ifd, FDR|FDW);
		hxd_fd_del(ifd);
		memset(&hxd_files[ifd], 0, sizeof(struct hxd_file));
	}
#ifdef CONFIG_IPV6
	inet_ntop(AFINET, (char *)&htlc->sockaddr.SIN_ADDR, buf, sizeof(buf));
#else
	inet_ntoa_r(htlc->sockaddr.SIN_ADDR, buf, sizeof(buf));
#endif
	hxd_log("%s@%s:%u - %s:%u:%u:%s - htlc connection closed",
		htlc->userid, buf, ntohs(htlc->sockaddr.SIN_PORT),
		htlc->name, htlc->icon, htlc->uid, htlc->login);
#if defined(CONFIG_SQL)
	sql_delete_user(htlc->userid, htlc->name, buf, ntohs(htlc->sockaddr.SIN_PORT), htlc->login, htlc->uid);
#endif
	timer_delete_ptr(htlc);
	can_login = htlc->access_extra.can_login;
	if (!can_login) {
		if (htlc->next)
			htlc->next->prev = htlc->prev;
		if (htlc->prev)
			htlc->prev->next = htlc->next;
		if (htlc_tail == htlc)
			htlc_tail = htlc->prev;
		chat_remove_from_all(htlc);
		uid16 = htons(mangle_uid(htlc));
		for (htlcp = htlc_list->next; htlcp; htlcp = htlcp->next) {
			if (!htlcp->access_extra.user_getlist)
				continue;
			hlwrite(htlcp, HTLS_HDR_USER_PART, 0, 1,
				HTLS_DATA_UID, sizeof(uid16), &uid16);
		}
	}
	if (htlc->read_in.buf)
		xfree(htlc->read_in.buf);
	if (htlc->in.buf)
		xfree(htlc->in.buf);
	if (htlc->out.buf)
		xfree(htlc->out.buf);
	LOCK_HTXF(htlc);
	for (i = 0; i < HTXF_PUT_MAX; i++) {
		if (htlc->htxf_in[i]) {
			wr = 0;
#if defined(CONFIG_HTXF_PTHREAD)
			if (htlc->htxf_in[i]->tid) {
				pthread_kill(htlc->htxf_in[i]->tid, SIGTERM);
				wr = htlc->htxf_in[i]->tid;
				xxx_remove(htlc->htxf_in[i]);
			}
#elif defined(CONFIG_HTXF_CLONE)
			mask_signal(SIG_BLOCK, SIGCHLD);
			if (htlc->htxf_in[i]->pid) {
				kill(htlc->htxf_in[i]->pid, SIGTERM);
				wr = waitpid(htlc->htxf_in[i]->pid, &status, 0);
				xxx_remove(htlc->htxf_in[i]);
			}
			mask_signal(SIG_UNBLOCK, SIGCHLD);
			if (htlc->htxf_in[i]->stack)
				xfree(htlc->htxf_in[i]->stack);
			xfree(htlc->htxf_in[i]);
#elif defined(CONFIG_HTXF_FORK)
			mask_signal(SIG_BLOCK, SIGCHLD);
			if (htlc->htxf_in[i]->pid) {
				kill(htlc->htxf_in[i]->pid, SIGTERM);
				wr = waitpid(htlc->htxf_in[i]->pid, &status, 0);
			}
			mask_signal(SIG_UNBLOCK, SIGCHLD);
			xfree(htlc->htxf_in[i]);
#endif
			if (wr != 0 && wr != -1)
				nr_puts--;
		}
	}
	for (i = 0; i < HTXF_GET_MAX; i++) {
		if (htlc->htxf_out[i]) {
			wr = 0;
#if defined(CONFIG_HTXF_PTHREAD)
			if (htlc->htxf_out[i]->tid) {
				pthread_kill(htlc->htxf_out[i]->tid, SIGTERM);
				wr = htlc->htxf_out[i]->tid;
				xxx_remove(htlc->htxf_out[i]);
			}
#elif defined(CONFIG_HTXF_CLONE)
			mask_signal(SIG_BLOCK, SIGCHLD);
			if (htlc->htxf_out[i]->pid) {
				kill(htlc->htxf_out[i]->pid, SIGTERM);
				wr = waitpid(htlc->htxf_out[i]->pid, &status, 0);
				xxx_remove(htlc->htxf_out[i]);
			}
			mask_signal(SIG_UNBLOCK, SIGCHLD);
			if (htlc->htxf_out[i]->stack)
				xfree(htlc->htxf_out[i]->stack);
			xfree(htlc->htxf_out[i]);
#elif defined(CONFIG_HTXF_FORK)
			mask_signal(SIG_BLOCK, SIGCHLD);
			if (htlc->htxf_out[i]->pid) {
				kill(htlc->htxf_out[i]->pid, SIGTERM);
				wr = waitpid(htlc->htxf_out[i]->pid, &status, 0);
			}
			mask_signal(SIG_UNBLOCK, SIGCHLD);
			xfree(htlc->htxf_out[i]);
#endif
			if (wr != 0 && wr != -1)
				nr_gets--;
		}
	}
	UNLOCK_HTXF(htlc);
#ifdef CONFIG_COMPRESS
	if (htlc->compress_encode_type != COMPRESS_NONE)
		compress_encode_end(htlc);
	if (htlc->compress_decode_type != COMPRESS_NONE)
		compress_decode_end(htlc);
#endif
#ifdef CONFIG_EXEC
	exec_close_all(htlc);
#endif
	xfree(htlc);
	if (!can_login) {
		nhtlc_conns--;
#ifdef CONFIG_TRACKER_REGISTER
		tracker_register_timer(0);
#endif
	}
}

#if defined(CONFIG_HTXF_CLONE) || defined(CONFIG_HTXF_FORK)
static int
free_htxf (void *__arg)
{
	u_int32_t x = (u_int32_t)__arg;
	u_int16_t i = (x >> 24) & 0x7f, get = x>>31;
	int fd = x & 0xffffff;
	struct htlc_conn *htlc;

	htlc = hxd_files[fd].conn.htlc;
	if (!htlc)
		return 0;
	LOCK_HTXF(htlc);
	if (get) {
		if (!htlc->htxf_out[i])
			return 0;
#if defined(CONFIG_HTXF_CLONE)
		xfree(htlc->htxf_out[i]->stack);
#else
#if !HTXF_THREADS_LISTEN
		htxf_close(htlc->htxf_out[i]->fd);
#endif
#endif
#if defined(CONFIG_HTXF_CLONE)
		xxx_remove(htlc->htxf_out[i]);
#endif
		xfree(htlc->htxf_out[i]);
		htlc->htxf_out[i] = 0;
		htlc->nr_gets--;
		nr_gets--;
	} else {
		if (!htlc->htxf_in[i])
			return 0;
#if defined(CONFIG_HTXF_CLONE)
		xfree(htlc->htxf_in[i]->stack);
#else
#if !HTXF_THREADS_LISTEN
		htxf_close(htlc->htxf_in[i]->fd);
#endif
#endif
#if defined(CONFIG_HTXF_CLONE)
		xxx_remove(htlc->htxf_in[i]);
#endif
		xfree(htlc->htxf_in[i]);
		htlc->htxf_in[i] = 0;
		htlc->nr_puts--;
		nr_puts--;
	}
	UNLOCK_HTXF(htlc);

	return 0;
}
#endif

void
hlserver_reap_pid (pid_t pid, int status __attribute__((__unused__)))
{
#if defined(CONFIG_HTXF_CLONE) || defined(CONFIG_HTXF_FORK)
	struct htlc_conn *htlcp;
	u_int16_t i;
	u_int32_t x;

	for (htlcp = htlc_list->next; htlcp; htlcp = htlcp->next) {
		for (i = 0; i < HTXF_GET_MAX; i++) {
			if (htlcp->htxf_out[i]) {
				if (htlcp->htxf_out[i]->pid == pid) {
					x = htlcp->fd | (i<<24) | (1<<31);
					timer_add_secs(0, free_htxf, (void *)x);
					return;
				}
			}
		}
		for (i = 0; i < HTXF_PUT_MAX; i++) {
			if (htlcp->htxf_in[i]) {
				if (htlcp->htxf_in[i]->pid == pid) {
					x = htlcp->fd | (i<<24);
					timer_add_secs(0, free_htxf, (void *)x);
					return;
				}
			}
		}
	}
#endif
}

void
hotline_server_init (void)
{
	struct SOCKADDR_IN saddr;
	int x, r, i, listen_sock;
	char *host;

	memset(&__htlc_list, 0, sizeof(__htlc_list));
	for (i = 0; (host = hxd_cfg.options.addresses[i]); i++) {
#ifdef CONFIG_IPV6
		if (!inet_pton(AFINET, host, &saddr.SIN_ADDR)) {
#else
		if (!inet_aton(host, &saddr.SIN_ADDR)) {
#endif
			struct hostent *he = gethostbyname(host);
			if (he)
				memcpy(&saddr.SIN_ADDR, he->h_addr_list[0],
				       (size_t)he->h_length > sizeof(saddr.SIN_ADDR) ?
				       sizeof(saddr.SIN_ADDR) : (size_t)he->h_length);
			else {
				hxd_log("%s:%d: could not resolve hostname %s", __FILE__, __LINE__, host);
				exit(1);
			}
		}

		listen_sock = socket(AFINET, SOCK_STREAM, IPPROTO_TCP);
		if (listen_sock < 0) {
#if defined(__WIN32__)
			hxd_log("%s:%d: socket: WSA %d", __FILE__, __LINE__, listen_sock, WSAGetLastError());
#endif
			hxd_log("%s:%d: socket: %s", __FILE__, __LINE__, strerror(errno));
			exit(1);
		}
		nr_open_files++;
		if (nr_open_files >= hxd_open_max) {
			hxd_log("%s:%d: %d >= hxd_open_max (%d)", __FILE__, __LINE__, listen_sock, hxd_open_max);
			socket_close(listen_sock);
			nr_open_files--;
			exit(1);
		}
		x = 1;
		setsockopt(listen_sock, SOL_SOCKET, SO_REUSEADDR, (SETSOCKOPT_PTR_CAST_T)&x, sizeof(x));
		saddr.SIN_FAMILY = AFINET;
		saddr.SIN_PORT = htons(hxd_cfg.options.port);
#ifdef CONFIG_EUID
		if (hxd_cfg.options.port <= 1024) {
			r = seteuid(0);
			if (r)
				hxd_log("seteuid(0): %s", strerror(errno));
		}
#endif
		r = bind(listen_sock, (struct SOCKADDR *)&saddr, sizeof(saddr));
#ifdef CONFIG_EUID
		if (hxd_cfg.options.port <= 1024)
			seteuid(getuid());
#endif
		if (r < 0) {
#ifdef CONFIG_IPV6
			char abuf[HOSTLEN+1];
			inet_ntop(AFINET, (char *)&saddr.SIN_ADDR, abuf, sizeof(abuf));
#else
			char abuf[16];
			inet_ntoa_r(saddr.SIN_ADDR, abuf, sizeof(abuf));
#endif

			hxd_log("%s:%d: bind(%s:%u): %s", __FILE__, __LINE__, abuf, ntohs(saddr.SIN_PORT), strerror(errno));
			exit(1);
		}
		if (listen(listen_sock, 5) < 0) {
			hxd_log("%s:%d: listen: %s", __FILE__, __LINE__, strerror(errno));
			exit(1);
		}

		hxd_files[listen_sock].ready_read = listen_ready_read;
		FD_SET(listen_sock, &hxd_rfds);
		if (high_fd < listen_sock)
			high_fd = listen_sock;
		fd_closeonexec(listen_sock, 1);
		socket_blocking(listen_sock, 0);

		htxf_init(&saddr);
	}
}
